<?php

/**
 * RecordCopyPaste - Export/Import relational database records via JSON with deployment tracking
 *
 * Handles copying records with their relations across databases with same structure.
 * Supports identifier-based duplicate detection, ID mapping, flexible overrides, and deployment tracking.
 *
 * Export uses JSON template syntax directly in SQL.
 *
 * @author enured
 * @version 1.0
 */

namespace IMATHUZH\Qfq\Core\Helper;

use Exception;
use IMATHUZH\Qfq\Core\Database\Database;
use IMATHUZH\Qfq\Core\Evaluate;
use IMATHUZH\Qfq\Core\Store\Store;

class CopyPaste {

    private $db;
    private array $idMapping = [];
    private ?Store $store;
    private string|int|array|null|false $dbIndexData;
    private string|int|array|null|false $dbIndexQfq;
    private mixed $dbArray;
    private mixed $content;
    private ?int $ruleId = null;
    private bool $enableDeployTracking = false;

    // Valid update modes
    const UPDATE_ONCE = 'once';
    const UPDATE_ALWAYS = 'always';
    const UPDATE_SKIP_MODIFIED = 'skipModified';

    /**
     * @param mixed $content JSON template, rule names, or import data
     * @param Database|null $db Database instance
     * @param Store|null $store Store instance
     * @param bool $phpUnit PHPUnit test mode
     * @param bool $enableDeployTracking Enable DeployRef tracking (for deployments)
     * @throws \CodeException
     * @throws \UserFormException
     * @throws \DbException
     * @throws \UserReportException
     */
    public function __construct($content, $db = null, $store = null, $phpUnit = false, $enableDeployTracking = false) {
        if (defined('PHPUNIT_QFQ')) {
            $phpUnit = true;
        }

        if ($store === null) {
            $store = Store::getInstance('', $phpUnit);
        }

        $this->content = trim($content);
        $this->store = $store;
        $this->enableDeployTracking = $enableDeployTracking;

        if ($db === null) {
            $this->dbIndexData = $this->store->getVar(SYSTEM_DB_INDEX_DATA, STORE_SYSTEM);
            $this->dbIndexQfq = $this->store->getVar(SYSTEM_DB_INDEX_QFQ, STORE_SYSTEM);

            $this->dbArray[$this->dbIndexData] = new Database($this->dbIndexData);

            if ($this->dbIndexData != $this->dbIndexQfq) {
                $this->dbArray[$this->dbIndexQfq] = new Database($this->dbIndexQfq);
            }

            $db = $this->dbArray[$this->dbIndexData];
        }

        $this->db = $db;
    }

    // Only use for debuggin reasons
    private function debug($message) {
        $logFile = '/tmp/copypaste_debug.log';
        file_put_contents($logFile, date('Y-m-d H:i:s') . ' ' . $message . "\n", FILE_APPEND);
    }

    /**
     * Handle Export: Parse JSON template or rule names, fetch data, generate complete JSON
     *
     * @return string JSON string for clipboard
     * @throws \UserReportException
     * @throws \Exception
     */
    public function handleExport(): string {
        // Auto-detect: JSON template or rule names
        if (str_starts_with($this->content, '{')) {
            // It's a JSON template
            $template = json_decode($this->content, true);

            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new \UserReportException("Invalid JSON template: " . json_last_error_msg());
            }
        } else {
            // It's comma-separated rule names
            $ruleNames = array_map('trim', explode(',', $this->content));
            $template = $this->loadRules($ruleNames);
        }

        if (!isset($template['element']) || !is_array($template['element'])) {
            throw new \Exception("Invalid template: missing element array");
        }

        // Build result with version, timestamp, and optional ruleId
        $result = [
            'version' => '2.0',
            'exportedAt' => date('c'),
            'element' => []
        ];

        // Include ruleId only if deployment tracking is enabled
        if ($this->enableDeployTracking && $this->ruleId !== null) {
            $result['ruleId'] = $this->ruleId;
        }

        // Process each root element
        foreach ($template['element'] as $elementTemplate) {
            $processedElements = $this->processElementTemplate($elementTemplate, null);
            $result['element'] = array_merge($result['element'], $processedElements);
        }

        return json_encode($result, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    }

    /**
     * Handle Import: Parse JSON + overrides, check duplicates, insert/update with DeployRef tracking
     *
     * @param string|null $importOverrides JSON format: {"element":[{"table":"Person","data":[{"id":600}]}]}
     *                                      Optional fields: relation, identifier, condition, element
     * @return array Result with status and created/updated IDs
     * @throws Exception
     */
    public function handleImport(?string $importOverrides = null): array {
        $data = json_decode($this->content, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new Exception("Invalid JSON: " . json_last_error_msg());
        }

        if (!isset($data['element'])) {
            throw new Exception("Invalid JSON structure: missing element array");
        }

        // Extract ruleId from import data if present
        $importRuleId = $data['ruleId'] ?? null;

        // Parse import overrides
        $overrides = [];
        if ($importOverrides) {
            $overrides = $this->parseImportOverrides($importOverrides);
        }

        // Reset ID mapping for this import
        $this->idMapping = [];

        try {
            $results = [
                'success' => true,
                'imported' => [],
                'idMapping' => []
            ];

            // Process all root elements recursively
            foreach ($data['element'] as $element) {
                $importResult = $this->importElementRecursive($element, $overrides, $importRuleId);
                $results['imported'][] = $importResult;
            }

            $results['idMapping'] = $this->idMapping;

            return $results;

        } catch (Exception $e) {
            // sql() handles rollback internally
            throw $e;
        }
    }

    /**
     * Load and merge multiple rule definitions from DeployRule table
     *
     * @param array $ruleNames Array of rule names
     * @return array Merged template with all elements
     * @throws Exception
     */
    private function loadRules(array $ruleNames): array {
        if (empty($ruleNames)) {
            throw new Exception("No rule names provided");
        }

        $evaluate = new Evaluate($this->store, $this->db);

        $mergedTemplate = [
            'element' => []
        ];

        // Track first rule ID for export metadata
        $firstRuleId = null;

        foreach ($ruleNames as $ruleName) {
            if (empty(trim($ruleName))) continue;

            $query = "SELECT id, definition FROM DeployRule WHERE name = ? LIMIT 1";
            $result = $this->db->sql($query, ROW_EXPECT_GE_1, [$ruleName]);

            if (empty($result)) {
                throw new Exception("Rule not found: {$ruleName}");
            }

            $row = $result[0];

            // Store first rule ID
            if ($firstRuleId === null) {
                $firstRuleId = $row['id'];
            }

            // Parse rule definition
            $ruleDefinition = $row['definition'];
            $ruleDefinition = $evaluate->parse($ruleDefinition);

            $ruleTemplate = json_decode($ruleDefinition, true);

            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new Exception("Invalid JSON in rule '{$ruleName}': " . json_last_error_msg());
            }

            // Merge elements
            if (isset($ruleTemplate['element']) && is_array($ruleTemplate['element'])) {
                $mergedTemplate['element'] = array_merge(
                    $mergedTemplate['element'],
                    $ruleTemplate['element']
                );
            }
        }

        // Set ruleId for export metadata
        $this->ruleId = $firstRuleId;

        return $mergedTemplate;
    }

    /**
     * Process element template: fetch data and process child elements (BATCH-OPTIMIZED)
     *
     * @param array $template Element template with table, condition, etc.
     * @param array|null $parentData Parent record data for relation mapping
     * @return array Array of processed elements (can be multiple if condition matches multiple records)
     */
    private function processElementTemplate(array $template, ?array $parentData): array {
        $table = $template['table'] ?? null;
        $dataTemplate = $template['data'] ?? null;
        $condition = $template['condition'] ?? null;
        $relation = $template['relation'] ?? null;
        $identifier = $template['identifier'] ?? null;
        $update = $template['update'] ?? self::UPDATE_ONCE;
        $keepId = $template['keepId'] ?? false;
        $childTemplates = $template['element'] ?? [];

        if (!$table) {
            throw new Exception("Element template missing table");
        }

        // Normalize update mode
        $update = $this->normalizeUpdateMode($update);

        // Fetch records for this level
        $records = $this->fetchRecordsForTemplate($table, $dataTemplate, $condition, $relation, $parentData);

        if (empty($records)) {
            return [];
        }

        // Process child elements in BATCHES
        $childElementsByTemplate = [];

        if (!empty($childTemplates)) {
            // Group child templates by (table, relation, condition) for batching
            foreach ($childTemplates as $childIndex => $childTemplate) {
                $batchKey = $this->getBatchKey($childTemplate);

                // Fetch all children for this template in ONE batch query
                // PASS PARENT TABLE NAME so fetchChildrenBatch can distinguish relation types
                $allChildRecords = $this->fetchChildrenBatch($childTemplate, $records, $table);

                // Group children by parent ID
                $childrenByParent = $this->groupChildrenByParent($allChildRecords, $childTemplate['relation'] ?? null);

                $childElementsByTemplate[$childIndex] = $childrenByParent;
            }
        }

        // Build elements with their children
        $elements = [];

        foreach ($records as $record) {
            $element = [
                'table' => $table,
                'data' => $record,
                'relation' => $relation,
                'identifier' => $identifier,
                'condition' => $condition,
                'update' => $update,
                'keepId' => $keepId,
                'element' => []
            ];

            // Assign pre-fetched children to this parent
            foreach ($childTemplates as $childIndex => $childTemplate) {
                $childRelation = $childTemplate['relation'] ?? null;

                // Try ALL relation keys - we don't know which one was used for grouping
                $childRecords = [];
                if ($childRelation) {
                    foreach ($childRelation as $childCol => $parentRef) {
                        [$refTable, $refCol] = explode('.', $parentRef, 2);

                        // Try to find children using this parent column
                        if (isset($record[$refCol])) {
                            $lookupKey = $record[$refCol];
                            if (isset($childElementsByTemplate[$childIndex][$lookupKey])) {
                                $childRecords = array_merge($childRecords, $childElementsByTemplate[$childIndex][$lookupKey]);
                            }
                        }
                    }

                    // Remove duplicates (same child might be found via multiple relations)
                    $childRecords = array_values(array_unique($childRecords, SORT_REGULAR));
                }

                if (!empty($childRecords)) {
                    // Recursively process these children's children
                    foreach ($childRecords as $childRecord) {
                        $processedChildren = $this->processChildElementsRecursive(
                            $childTemplate,
                            $childRecord
                        );
                        $element['element'] = array_merge($element['element'], $processedChildren);
                    }
                }
            }

            $elements[] = $element;
        }

        // If multiple records with same template, group data into array
        if (count($elements) > 1 && $this->shouldGroupData($elements)) {
            // Collect all unique child elements from all parents
            $allChildElements = [];
            $seenChildKeys = [];

            foreach ($elements as $element) {
                foreach ($element['element'] as $childElement) {
                    // Create unique key for deduplication (table + data identifier)
                    $childTable = $childElement['table'];
                    $childIdentifier = $childElement['identifier'] ?? null;

                    // Use identifier fields or all data for uniqueness check
                    if ($childIdentifier && $childIdentifier !== '') {
                        if ($childIdentifier === '*') {
                            // Use all non-id fields
                            $keyData = $childElement['data'];
                            unset($keyData['id'], $keyData['created'], $keyData['modified']);
                        } else {
                            // Use specified identifier fields
                            $identifierFields = explode('-', $childIdentifier);
                            $keyData = [];
                            foreach ($identifierFields as $field) {
                                $keyData[$field] = $childElement['data'][$field] ?? null;
                            }
                        }
                    } else {
                        // No identifier - use id or all data
                        $keyData = ['id' => $childElement['data']['id'] ?? json_encode($childElement['data'])];
                    }

                    $uniqueKey = $childTable . ':' . json_encode($keyData);

                    // Only add if not seen before
                    if (!isset($seenChildKeys[$uniqueKey])) {
                        $allChildElements[] = $childElement;
                        $seenChildKeys[$uniqueKey] = true;
                    }
                }
            }

            // Group child elements if they share same table/relation/identifier/condition
            $allChildElements = $this->groupChildElements($allChildElements);

            $grouped = [
                'table' => $table,
                'data' => array_map(fn($e) => $e['data'], $elements),
                'relation' => $relation,
                'identifier' => $identifier,
                'condition' => $condition,
                'update' => $update,
                'keepId' => $keepId,
                'element' => $allChildElements
            ];

            return [$grouped];
        }

        // Single parent or ungroupable parents - group children for each
        foreach ($elements as &$element) {
            if (!empty($element['element'])) {
                $element['element'] = $this->groupChildElements($element['element']);
            }
        }

        return $elements;
    }

    /**
     * Normalize update mode - convert old boolean format to new string format
     *
     * @param mixed $update Update mode (boolean or string)
     * @return string Normalized update mode
     */
    private function normalizeUpdateMode($update): string {
        // Handle old boolean format for backwards compatibility
        if (is_bool($update)) {
            return $update ? self::UPDATE_ALWAYS : self::UPDATE_ONCE;
        }

        // Validate string format
        if (in_array($update, [self::UPDATE_ONCE, self::UPDATE_ALWAYS, self::UPDATE_SKIP_MODIFIED])) {
            return $update;
        }

        // Default to 'once' for invalid values
        return self::UPDATE_ONCE;
    }

    /**
     * Fetch records for a template
     */
    private function fetchRecordsForTemplate(string $table, ?array $dataTemplate, ?string $condition, ?array $relation, ?array $parentData): array {
        $whereConditions = [];
        $params = [];

        // Handle relation (FK condition)
        if ($relation && $parentData) {
            foreach ($relation as $childCol => $parentRef) {
                [$parentTable, $parentCol] = explode('.', $parentRef, 2);

                // Determine which parent value to use based on relation type
                $parentValue = null;

                if ($childCol === 'id') {
                    // Reverse relation: use parentData[parentCol]
                    $parentValue = $parentData[$parentCol] ?? null;
                }
                elseif ($parentTable === $table) {
                    // Self-referencing: use parentData[parentCol] (usually 'id')
                    $parentValue = $parentData[$parentCol] ?? null;
                }
                elseif (isset($parentData[$childCol]) && $parentData[$childCol] !== null && $parentData[$childCol] !== 0 && $parentData[$childCol] !== '0') {
                    // Cross-table hierarchy: parent has same FK as child
                    $parentValue = $parentData[$childCol];
                }
                else {
                    // Normal relation: use parentData[parentCol]
                    $parentValue = $parentData[$parentCol] ?? null;
                }

                if ($parentValue !== null) {
                    $whereConditions[] = $this->escapeIdentifier($childCol) . " = ?";
                    $params[] = $parentValue;
                }
            }
        }

        // Handle IDs from data template
        if ($dataTemplate) {
            // Check if it's a single record (associative array) or array of records (sequential array)
            if (!isset($dataTemplate[0])) {
                // Single record - wrap it in array
                $dataTemplate = [$dataTemplate];
            }

            if (isset($dataTemplate[0]['id'])) {
                $ids = array_map(fn($d) => $d['id'], $dataTemplate);
                $ids = array_filter($ids); // Remove nulls

                if (!empty($ids)) {
                    $placeholders = implode(',', array_fill(0, count($ids), '?'));
                    $whereConditions[] = $this->escapeIdentifier('id') . " IN ({$placeholders})";
                    foreach ($ids as $id) {
                        $params[] = $id;
                    }
                }
            }
        }

        // Handle condition SQL string
        if ($condition) {
            $whereConditions[] = "({$condition})";
        }

        $query = "SELECT * FROM " . $this->escapeIdentifier($table);
        if (!empty($whereConditions)) {
            $query .= " WHERE " . implode(' AND ', $whereConditions);
        }

        $result = $this->db->sql($query, ROW_REGULAR, $params);

        return $result;
    }

    /**
     * Fetch children for multiple parents in ONE batch query
     *
     * @param array $childTemplate Child element template
     * @param array $parentRecords Parent records
     * @param string $parentTable Parent table name (needed to distinguish relation types)
     * @return array Child records
     */
    private function fetchChildrenBatch(array $childTemplate, array $parentRecords, string $parentTable): array {
        if (empty($parentRecords)) {
            return [];
        }

        $table = $childTemplate['table'] ?? null;
        $condition = $childTemplate['condition'] ?? null;
        $relation = $childTemplate['relation'] ?? null;

        if (!$table || !$relation) {
            return [];
        }

        // Collect parent values for each relation
        $whereConditions = [];
        $params = [];

        foreach ($relation as $childCol => $parentRef) {
            [$refTable, $refCol] = explode('.', $parentRef, 2);

            // Determine relation type and which column to use
            $relationIds = [];
            foreach ($parentRecords as $parent) {
                $value = null;

                // Check 1: Reverse relation (child.id references parent.fkCol)
                if ($childCol === 'id') {
                    if (isset($parent[$refCol]) && $parent[$refCol] !== null && $parent[$refCol] !== 0 && $parent[$refCol] !== '0') {
                        $value = $parent[$refCol];
                    }
                }
                // Check 2: Self-referencing (child and parent are same table)
                elseif ($refTable === $table) {
                    if (isset($parent[$refCol]) && $parent[$refCol] !== null && $parent[$refCol] !== 0 && $parent[$refCol] !== '0') {
                        $value = $parent[$refCol];
                    }
                }
                // Check 3: CROSS-TABLE HIERARCHY
                elseif ($parentTable !== $refTable && isset($parent[$childCol]) && $parent[$childCol] !== null && $parent[$childCol] !== 0 && $parent[$childCol] !== '0') {
                    $value = $parent[$childCol];
                }
                // Check 4: NORMAL RELATION
                elseif (isset($parent[$refCol]) && $parent[$refCol] !== null && $parent[$refCol] !== 0 && $parent[$refCol] !== '0') {
                    $value = $parent[$refCol];
                }

                if ($value !== null) {
                    $relationIds[] = $value;
                }
            }

            $relationIds = array_unique($relationIds);

            if (!empty($relationIds)) {
                $placeholders = implode(',', array_fill(0, count($relationIds), '?'));
                $whereConditions[] = $this->escapeIdentifier($childCol) . " IN ({$placeholders})";
                foreach ($relationIds as $id) {
                    $params[] = $id;
                }
            }
        }

        if (empty($whereConditions)) {
            return [];
        }

        // Combine multiple relation conditions with AND (all must match)
        $relationCondition = '(' . implode(' AND ', $whereConditions) . ')';

        // Additional condition from template
        $finalConditions = [$relationCondition];
        if ($condition) {
            $finalConditions[] = "({$condition})";
        }

        $query = "SELECT * FROM " . $this->escapeIdentifier($table);
        $query .= " WHERE " . implode(' AND ', $finalConditions);

        $result = $this->db->sql($query, ROW_REGULAR, $params);

        return $result;
    }

    /**
     * Group child records by parent ID
     */
    private function groupChildrenByParent(array $childRecords, ?array $relation): array {
        if (empty($childRecords) || !$relation) {
            return [];
        }

        $grouped = [];

        // Try each FK column and choose the one with most unique values
        $bestColumn = null;
        $maxUniqueValues = 0;

        foreach ($relation as $childCol => $parentRef) {
            [$refTable, $refCol] = explode('.', $parentRef, 2);

            // Count unique non-zero values for this FK column
            $uniqueValues = [];
            foreach ($childRecords as $record) {
                $value = $record[$childCol] ?? null;
                if ($value !== null && $value !== 0 && $value !== '0') {
                    $uniqueValues[$value] = true;
                }
            }

            $uniqueCount = count($uniqueValues);

            if ($uniqueCount > $maxUniqueValues || ($uniqueCount === $maxUniqueValues && $uniqueCount > 0)) {
                $maxUniqueValues = $uniqueCount;
                $bestColumn = $childCol;
            }
        }

        // If no suitable column found, return empty
        if ($bestColumn === null) {
            return [];
        }

        // Group by the best FK column
        foreach ($childRecords as $record) {
            $parentId = $record[$bestColumn] ?? null;
            if ($parentId !== null && $parentId !== 0 && $parentId !== '0') {
                if (!isset($grouped[$parentId])) {
                    $grouped[$parentId] = [];
                }
                $grouped[$parentId][] = $record;
            }
        }

        return $grouped;
    }

    /**
     * Process child elements recursively (for nested children)
     */
    private function processChildElementsRecursive(array $template, array $recordData): array {
        $childTemplates = $template['element'] ?? [];

        $element = [
            'table' => $template['table'],
            'data' => $recordData,
            'relation' => $template['relation'] ?? null,
            'identifier' => $template['identifier'] ?? null,
            'condition' => $template['condition'] ?? null,
            'update' => $this->normalizeUpdateMode($template['update'] ?? self::UPDATE_ONCE),
            'keepId' => $template['keepId'] ?? false,
            'element' => []
        ];

        // Process nested children (recursive)
        if (!empty($childTemplates)) {
            foreach ($childTemplates as $childTemplate) {
                $processedChildren = $this->processElementTemplate($childTemplate, $recordData);
                $element['element'] = array_merge($element['element'], $processedChildren);
            }
        }

        return [$element];
    }

    /**
     * Get batch key for grouping child templates
     */
    private function getBatchKey(array $template): string {
        return json_encode([
            'table' => $template['table'] ?? null,
            'relation' => $template['relation'] ?? null,
            'condition' => $template['condition'] ?? null
        ]);
    }

    /**
     * Determine if multiple elements should be grouped with array data
     */
    private function shouldGroupData(array $elements): bool {
        if (count($elements) <= 1) {
            return false;
        }

        // Don't group if any element has children
        foreach ($elements as $element) {
            if (!empty($element['element'])) {
                return false;
            }
        }

        // Check if all elements have same metadata
        $first = $elements[0];
        $firstRelation = json_encode($first['relation']);
        $firstIdentifier = $first['identifier'];
        $firstCondition = $first['condition'];
        $firstUpdate = $first['update'] ?? self::UPDATE_ONCE;
        $firstKeepId = $first['keepId'] ?? false;

        foreach ($elements as $element) {
            if (json_encode($element['relation']) !== $firstRelation ||
                $element['identifier'] !== $firstIdentifier ||
                $element['condition'] !== $firstCondition ||
                ($element['update'] ?? self::UPDATE_ONCE) !== $firstUpdate ||
                ($element['keepId'] ?? false) !== $firstKeepId) {
                return false;
            }
        }

        return true;
    }

    /**
     * Group child elements that have the same table, relation, identifier, and condition
     */
    private function groupChildElements(array $childElements): array {
        if (empty($childElements)) {
            return [];
        }

        // Group by table + relation + identifier + condition + update
        $groups = [];

        foreach ($childElements as $childElement) {
            $groupKey = json_encode([
                'table' => $childElement['table'],
                'relation' => $childElement['relation'] ?? null,
                'identifier' => $childElement['identifier'] ?? null,
                'condition' => $childElement['condition'] ?? null,
                'update' => $childElement['update'] ?? self::UPDATE_ONCE,
                'keepId' => $childElement['keepId'] ?? false
            ]);

            if (!isset($groups[$groupKey])) {
                $groups[$groupKey] = [
                    'template' => $childElement,
                    'data' => [],
                    'childElements' => []
                ];
            }

            $groups[$groupKey]['data'][] = $childElement['data'];

            // Collect grandchildren (nested elements) for deduplication
            if (!empty($childElement['element'])) {
                foreach ($childElement['element'] as $grandchild) {
                    $groups[$groupKey]['childElements'][] = $grandchild;
                }
            }
        }

        // Build result
        $result = [];

        foreach ($groups as $group) {
            $template = $group['template'];
            $dataRecords = $group['data'];
            $grandchildren = $group['childElements'];

            // Flatten nested data arrays
            $flatDataRecords = [];
            foreach ($dataRecords as $dataRecord) {
                if (is_array($dataRecord) && isset($dataRecord[0]) && is_array($dataRecord[0])) {
                    foreach ($dataRecord as $nestedRecord) {
                        $flatDataRecords[] = $nestedRecord;
                    }
                } else {
                    $flatDataRecords[] = $dataRecord;
                }
            }

            $template['data'] = count($flatDataRecords) === 1
                ? $flatDataRecords[0]
                : $flatDataRecords;

            // Deduplicate and group grandchildren
            if (!empty($grandchildren)) {
                $uniqueGrandchildren = [];
                $seenKeys = [];
                foreach ($grandchildren as $grandchild) {
                    $childTable = $grandchild['table'];
                    $childIdentifier = $grandchild['identifier'] ?? null;
                    $grandchildData = $grandchild['data'];

                    // Check if data is already grouped (array of records)
                    $isGroupedData = is_array($grandchildData) && isset($grandchildData[0]) && is_array($grandchildData[0]);

                    if ($isGroupedData) {
                        // For grouped data, build key from all records' identifiers
                        $allKeys = [];
                        foreach ($grandchildData as $record) {
                            if ($childIdentifier === '*') {
                                $recordKey = $record;
                                unset($recordKey['id'], $recordKey['created'], $recordKey['modified']);
                            } elseif ($childIdentifier && $childIdentifier !== '') {
                                $identifierFields = explode('-', $childIdentifier);
                                $recordKey = [];
                                foreach ($identifierFields as $field) {
                                    $recordKey[$field] = $record[$field] ?? null;
                                }
                            } else {
                                $recordKey = ['id' => $record['id'] ?? null];
                            }
                            $allKeys[] = $recordKey;
                        }
                        $keyData = $allKeys;
                    } elseif ($childIdentifier && $childIdentifier !== '') {
                        if ($childIdentifier === '*') {
                            $keyData = $grandchildData;
                            unset($keyData['id'], $keyData['created'], $keyData['modified']);
                        } else {
                            $identifierFields = explode('-', $childIdentifier);
                            $keyData = [];
                            foreach ($identifierFields as $field) {
                                $keyData[$field] = $grandchildData[$field] ?? null;
                            }
                        }
                    } else {
                        $keyData = ['id' => $grandchildData['id'] ?? json_encode($grandchildData)];
                    }

                    $uniqueKey = $childTable . ':' . json_encode($keyData);
                    if (!isset($seenKeys[$uniqueKey])) {
                        $uniqueGrandchildren[] = $grandchild;
                        $seenKeys[$uniqueKey] = true;
                    }
                }
                $template['element'] = $this->groupChildElements($uniqueGrandchildren);
            }

            $result[] = $template;
        }

        return $result;
    }

    /**
     * Parse import overrides from JSON structure
     */
    private function parseImportOverrides(string $definition): array {
        $overrideData = json_decode($definition, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new Exception("Invalid override JSON: " . json_last_error_msg());
        }

        if (!isset($overrideData['element'])) {
            throw new Exception("Invalid override structure: missing element array");
        }

        return $this->flattenOverrideElements($overrideData['element']);
    }

    /**
     * Flatten override elements recursively into table => data mapping
     */
    private function flattenOverrideElements(array $elements): array {
        $result = [];

        foreach ($elements as $element) {
            $table = $element['table'] ?? null;

            if (!$table) {
                continue;
            }

            $data = $element['data'] ?? [];

            // Handle both single object and array of data
            if (isset($data[0]) && is_array($data[0])) {
                $mergedData = [];
                foreach ($data as $dataRecord) {
                    $mergedData = array_merge($mergedData, $dataRecord);
                }
                $data = $mergedData;
            }

            if (!isset($result[$table])) {
                $result[$table] = [];
            }

            $result[$table] = array_merge($result[$table], $data);

            // Process nested elements recursively
            if (isset($element['element']) && is_array($element['element'])) {
                $childOverrides = $this->flattenOverrideElements($element['element']);

                foreach ($childOverrides as $childTable => $childData) {
                    if (!isset($result[$childTable])) {
                        $result[$childTable] = [];
                    }
                    $result[$childTable] = array_merge($result[$childTable], $childData);
                }
            }
        }

        return $result;
    }

    /**
     * Import element recursively with DeployRef tracking
     */
    private function importElementRecursive(array $element, array $overrides, ?int $ruleId = null): array {
        $table = $element['table'];
        $dataInput = $element['data'];
        $identifier = $element['identifier'] ?? null;
        $relation = $element['relation'] ?? [];
        $updateMode = $this->normalizeUpdateMode($element['update'] ?? self::UPDATE_ONCE);
        $keepId = $element['keepId'] ?? false;

        $result = [
            'table' => $table,
            'items' => [],
        ];

        // Handle data as array or single object
        if (isset($dataInput[0]) && is_array($dataInput[0])) {
            $dataArray = $dataInput;
        } else {
            $dataArray = [$dataInput];
        }

        // Check if this element has child relations (parent->child FKs)
        $hasChildRelations = false;
        if (!empty($dataArray)) {
            $firstData = $dataArray[0];
            if (isset($overrides[$table])) {
                $firstData = array_merge($firstData, $overrides[$table]);
            }
            $hasChildRelations = $this->hasChildRelations($firstData, $element['element'] ?? []);
        }

        // Collect child results
        $childResults = [];

        if ($hasChildRelations) {
            // CHILD-FIRST STRATEGY
            $childrenImported = false;

            foreach ($dataArray as $dataIndex => $data) {
                $exportId = $data['id'] ?? null;
                $exportModified = $data['modified'] ?? null;

                // Apply overrides
                if (isset($overrides[$table])) {
                    $data = array_merge($data, $overrides[$table]);
                }

                $oldId = $exportId;

                // Map PARENT relation keys
                $data = $this->mapRelationKeys($data, $relation);

                $itemResult = [
                    'action' => null,
                    'oldId' => $oldId,
                    'newId' => null,
                ];

                // Import children first (for child relations)
                if (!$childrenImported && !empty($element['element'])) {
                    foreach ($element['element'] as $childElement) {
                        $childResult = $this->importElementRecursive($childElement, $overrides, $ruleId);
                        $childResults[] = $childResult;
                    }
                    $childrenImported = true;
                }

                // Map child relation keys
                $data = $this->mapChildRelationKeys($data, $element['element'] ?? []);

                // Remove id from data (will be handled by insert/update logic)
                unset($data['id']);

                // Apply deployment logic
                $deployResult = $this->applyDeploymentLogic(
                    $table,
                    $data,
                    $identifier,
                    $relation,
                    $updateMode,
                    $keepId,
                    $oldId,
                    $exportModified,
                    $ruleId
                );

                $itemResult['action'] = $deployResult['action'];
                $itemResult['newId'] = $deployResult['newId'];

                // Update ID mapping
                if ($oldId && $itemResult['newId']) {
                    $this->idMapping[$table][$oldId] = $itemResult['newId'];
                }

                $result['items'][] = $itemResult;
            }

        } else {
            // PARENT-FIRST STRATEGY
            foreach ($dataArray as $dataIndex => $data) {
                $exportId = $data['id'] ?? null;
                $exportModified = $data['modified'] ?? null;

                // Apply overrides
                if (isset($overrides[$table])) {
                    $data = array_merge($data, $overrides[$table]);
                }

                $oldId = $exportId;

                // Map PARENT relation keys
                $data = $this->mapRelationKeys($data, $relation);

                // Remove id from data
                unset($data['id']);

                $itemResult = [
                    'action' => null,
                    'oldId' => $oldId,
                    'newId' => null,
                ];

                // Apply deployment logic
                $deployResult = $this->applyDeploymentLogic(
                    $table,
                    $data,
                    $identifier,
                    $relation,
                    $updateMode,
                    $keepId,
                    $oldId,
                    $exportModified,
                    $ruleId
                );

                $itemResult['action'] = $deployResult['action'];
                $itemResult['newId'] = $deployResult['newId'];

                // Update ID mapping
                if ($oldId && $itemResult['newId']) {
                    $this->idMapping[$table][$oldId] = $itemResult['newId'];
                }

                $result['items'][] = $itemResult;
            }

            // Import children after parents
            if (!empty($element['element'])) {
                foreach ($element['element'] as $childElement) {
                    $childResult = $this->importElementRecursive($childElement, $overrides, $ruleId);
                    $childResults[] = $childResult;
                }
            }
        }

        // Add child results
        if (!empty($childResults)) {
            $result['children'] = $childResults;
        }

        return $result;
    }

    /**
     * Apply deployment logic based on DeployRef, identifier, and update mode
     *
     * Implements the complex matrix logic for insert/update/skip decisions
     *
     * @param string $table Table name
     * @param array $data Record data (without id)
     * @param string|null $identifier Identifier columns
     * @param array $relation Relation mapping
     * @param string $updateMode Update mode (once, always, skipModified)
     * @param bool $keepId Whether to try keeping original ID
     * @param int|null $srcId Source record ID from export
     * @param string|null $exportModified Modified timestamp from export
     * @param int|null $ruleId DeployRule ID
     * @return array ['action' => string, 'newId' => int]
     * @throws Exception
     */
    private function applyDeploymentLogic(
        string $table,
        array $data,
        ?string $identifier,
        array $relation,
        string $updateMode,
        bool $keepId,
        ?int $srcId,
        ?string $exportModified,
        ?int $ruleId
    ): array {
        // Check if DeployRef entry exists for this source record (only if tracking enabled)
        $deployRef = null;
        if ($this->enableDeployTracking && $srcId && $ruleId) {
            $deployRef = $this->getDeployRef($table, $srcId, $ruleId);
        }

        // Find existing records by identifier
        $existingRecords = [];
        if ($identifier !== null && $identifier !== '') {
            $existingRecords = $this->findByIdentifierMulti($table, $data, $identifier, $relation);
        }

        $foundCount = count($existingRecords);
        $targetId = $foundCount === 1 ? $existingRecords[0]['id'] : null;
        $targetModified = $foundCount === 1 ? ($existingRecords[0]['modified'] ?? null) : null;

        // Apply matrix logic
        $action = null;
        $newId = null;

        // Case 1: DeployRef exists (srcId > 0)
        if ($deployRef) {
            $deployTargetId = $deployRef['targetId'];
            $deployModified = $deployRef['modified'];

            if ($identifier === null || $identifier === '') {
                // Identifier undefined -> always insert
                $preferredId = ($keepId && $srcId) ? $srcId : null;
                $newId = $this->insertRecord($table, $data, $preferredId);
                $action = 'inserted';

                // Update DeployRef with new targetId
                if ($this->enableDeployTracking) {
                    $this->updateDeployRef($deployRef['id'], $newId, $data['modified'] ?? date('Y-m-d H:i:s'));
                }

            } else {
                // Identifier defined
                if ($foundCount === 0) {
                    // Not found -> no changes (record was deleted in target)
                    $action = 'skipped';
                    $newId = $deployTargetId;

                } elseif ($foundCount === 1) {
                    // Found exactly one
                    if ($updateMode === self::UPDATE_ONCE) {
                        // once -> no changes
                        $action = 'existing';
                        $newId = $targetId;

                    } elseif ($updateMode === self::UPDATE_ALWAYS) {
                        // always -> update
                        $this->updateRecord($table, $targetId, $data);
                        $action = 'updated';
                        $newId = $targetId;

                        // Update DeployRef
                        if ($this->enableDeployTracking) {
                            $this->updateDeployRef($deployRef['id'], $newId, $data['modified'] ?? date('Y-m-d H:i:s'));
                        }

                    } elseif ($updateMode === self::UPDATE_SKIP_MODIFIED) {
                        // skipModified -> compare timestamps
                        if ($this->isSourceNewer($exportModified, $targetModified)) {
                            $this->updateRecord($table, $targetId, $data);
                            $action = 'updated';
                            $newId = $targetId;

                            // Update DeployRef
                            if ($this->enableDeployTracking) {
                                $this->updateDeployRef($deployRef['id'], $newId, $data['modified'] ?? date('Y-m-d H:i:s'));
                            }
                        } else {
                            $action = 'skipped';
                            $newId = $targetId;
                        }
                    }

                } else {
                    // Multiple records found -> exception
                    throw new Exception("Multiple records found for identifier in table {$table} (DeployRef exists)");
                }
            }

        } else {
            // Case 2: No DeployRef (srcId not exists or first deployment)
            if ($identifier === null || $identifier === '') {
                // Identifier undefined -> always insert
                $preferredId = ($keepId && $srcId) ? $srcId : null;
                $newId = $this->insertRecord($table, $data, $preferredId);
                $action = 'inserted';

                // Create DeployRef entry
                if ($this->enableDeployTracking && $srcId && $ruleId) {
                    $this->createDeployRef($table, $srcId, $newId, $data['modified'] ?? date('Y-m-d H:i:s'), $ruleId);
                }

            } else {
                // Identifier defined
                if ($foundCount === 0) {
                    // Not found -> insert
                    $preferredId = ($keepId && $srcId) ? $srcId : null;
                    $newId = $this->insertRecord($table, $data, $preferredId);
                    $action = 'inserted';

                    // Create DeployRef entry
                    if ($this->enableDeployTracking && $srcId && $ruleId) {
                        $this->createDeployRef($table, $srcId, $newId, $data['modified'] ?? date('Y-m-d H:i:s'), $ruleId);
                    }

                } elseif ($foundCount === 1) {
                    // Found exactly one
                    if ($updateMode === self::UPDATE_ONCE) {
                        // once -> link existing record in DeployRef
                        $action = 'existing';
                        $newId = $targetId;

                        // Create DeployRef entry (link to existing)
                        if ($this->enableDeployTracking && $srcId && $ruleId) {
                            $this->createDeployRef($table, $srcId, $newId, $targetModified ?? date('Y-m-d H:i:s'), $ruleId);
                        }

                    } elseif ($updateMode === self::UPDATE_ALWAYS) {
                        // always -> update and create DeployRef
                        $this->updateRecord($table, $targetId, $data);
                        $action = 'updated';
                        $newId = $targetId;

                        // Create DeployRef entry
                        if ($this->enableDeployTracking && $srcId && $ruleId) {
                            $this->createDeployRef($table, $srcId, $newId, $data['modified'] ?? date('Y-m-d H:i:s'), $ruleId);
                        }

                    } elseif ($updateMode === self::UPDATE_SKIP_MODIFIED) {
                        // skipModified -> compare timestamps
                        if ($this->isSourceNewer($exportModified, $targetModified)) {
                            $this->updateRecord($table, $targetId, $data);
                            $action = 'updated';
                            $newId = $targetId;
                        } else {
                            $action = 'skipped';
                            $newId = $targetId;
                        }

                        // Create DeployRef entry
                        if ($this->enableDeployTracking && $srcId && $ruleId) {
                            $this->createDeployRef($table, $srcId, $newId, $targetModified ?? date('Y-m-d H:i:s'), $ruleId);
                        }
                    }

                } else {
                    // Multiple records found -> exception
                    throw new Exception("Multiple records found for identifier in table {$table} (no DeployRef)");
                }
            }
        }

        return [
            'action' => $action,
            'newId' => $newId
        ];
    }

    /**
     * Get DeployRef entry for source record
     */
    private function getDeployRef(string $table, int $srcId, int $ruleId): ?array {
        $query = "SELECT id, targetId, modified FROM DeployRef 
                  WHERE tableName = ? AND srcId = ? AND ruleId = ? 
                  LIMIT 1";

        $result = $this->db->sql($query, ROW_REGULAR, [$table, $srcId, $ruleId]);

        return $result[0] ?? null;
    }

    /**
     * Create DeployRef entry
     */
    private function createDeployRef(string $table, int $srcId, int $targetId, string $modified, int $ruleId): void {
        $query = "INSERT INTO DeployRef (tableName, srcId, targetId, modified, ruleId) 
                  VALUES (?, ?, ?, ?, ?)";

        $this->db->sql($query, ROW_REGULAR, [$table, $srcId, $targetId, $modified, $ruleId]);
    }

    /**
     * Update DeployRef entry
     */
    private function updateDeployRef(int $deployRefId, int $targetId, string $modified): void {
        $query = "UPDATE DeployRef 
                  SET targetId = ?, modified = ? 
                  WHERE id = ?";

        $this->db->sql($query, ROW_REGULAR, [$targetId, $modified, $deployRefId]);
    }

    /**
     * Compare timestamps to determine if source is newer
     */
    private function isSourceNewer(?string $exportModified, ?string $targetModified): bool {
        if ($exportModified === null || $targetModified === null) {
            // If either timestamp is missing, consider source as newer
            return true;
        }

        $exportTime = strtotime($exportModified);
        $targetTime = strtotime($targetModified);

        return $exportTime > $targetTime;
    }

    /**
     * Find existing record by identifier columns (returns single result or null)
     */
    private function findByIdentifier(string $table, array $data, string $identifier, array $relation = []): ?int {
        $results = $this->findByIdentifierMulti($table, $data, $identifier, $relation);
        return $results[0]['id'] ?? null;
    }

    /**
     * Find existing records by identifier columns (returns array of matching records)
     */
    private function findByIdentifierMulti(string $table, array $data, string $identifier, array $relation = []): array {
        $tableName = $this->escapeIdentifier($table);

        // Determine which columns to use for identification
        $identifierColumns = [];
        if ($identifier === '*') {
            // Use all columns EXCEPT: id, created, modified, and relation FK columns
            $excludeColumns = ['id', 'created', 'modified'];

            // Add relation FK columns to exclude
            foreach ($relation as $fkColumn => $parentRef) {
                $excludeColumns[] = $fkColumn;
            }

            $identifierColumns = array_diff(array_keys($data), $excludeColumns);
        } else {
            // Use specified columns (separated by -)
            $identifierColumns = array_map('trim', explode('-', $identifier));
        }

        if (empty($identifierColumns)) {
            return [];
        }

        // Build WHERE clause
        $conditions = [];
        $params = [];

        foreach ($identifierColumns as $column) {
            if (!array_key_exists($column, $data)) {
                continue;
            }

            $escapedColumn = $this->escapeIdentifier($column);
            $value = $data[$column];

            if ($value === null) {
                $conditions[] = "{$escapedColumn} IS NULL";
            } else {
                $conditions[] = "{$escapedColumn} = ?";
                $params[] = $value;
            }
        }

        if (empty($conditions)) {
            return [];
        }

        $query = "SELECT id, modified FROM {$tableName} WHERE " . implode(' AND ', $conditions);

        $result = $this->db->sql($query, ROW_REGULAR, $params);

        return $result;
    }

    /**
     * Insert record into database
     */
    private function insertRecord(string $table, array $data, ?int $preferredId = null): int {
        if (empty($data)) {
            throw new Exception("Cannot insert empty data into {$table}");
        }

        $tableName = $this->escapeIdentifier($table);

        // Try to use preferred ID if provided and available
        if ($preferredId !== null && $this->isIdAvailable($table, $preferredId)) {
            $data['id'] = $preferredId;
        }

        $columns = array_keys($data);
        $values = array_values($data);

        $columnsList = implode(', ', array_map(fn($c) => $this->escapeIdentifier($c), $columns));
        $placeholders = implode(', ', array_fill(0, count($values), '?'));

        $query = "INSERT INTO {$tableName} ({$columnsList}) VALUES ({$placeholders})";

        $insertId = $this->db->sql($query, ROW_REGULAR, $values);

        if (!$insertId) {
            throw new Exception("Insert failed for {$table}: could not retrieve insert ID");
        }

        return $insertId;
    }

    /**
     * Update existing record in database
     */
    private function updateRecord(string $table, int $id, array $data): int {
        unset($data['id']);

        if (empty($data)) {
            return $id;
        }

        $tableName = $this->escapeIdentifier($table);
        $setClauses = [];
        $values = [];

        foreach ($data as $column => $value) {
            $setClauses[] = $this->escapeIdentifier($column) . ' = ?';
            $values[] = $value;
        }

        $values[] = $id;

        $query = "UPDATE {$tableName} SET " . implode(', ', $setClauses) . " WHERE id = ?";

        $this->db->sql($query, ROW_REGULAR, $values);

        return $id;
    }

    /**
     * Check if a specific ID is available (not used) in a table
     */
    private function isIdAvailable(string $table, int $id): bool {
        $tableName = $this->escapeIdentifier($table);

        $query = "SELECT COUNT(*) as count FROM {$tableName} WHERE id = ?";
        $result = $this->db->sql($query, ROW_REGULAR, [$id]);

        if (empty($result)) {
            return true;
        }

        return (int)$result[0]['count'] === 0;
    }

    /**
     * Map relation keys from old IDs to new IDs (PARENT relations)
     */
    private function mapRelationKeys(array $data, array $relation): array {
        $mapped = $data;

        foreach ($relation as $column => $reference) {
            [$referencesTable, $referencesColumn] = explode('.', $reference, 2);
            $oldValue = $data[$column] ?? null;

            if ($oldValue !== null && isset($this->idMapping[$referencesTable][$oldValue])) {
                $mapped[$column] = $this->idMapping[$referencesTable][$oldValue];
            }
        }

        return $mapped;
    }

    /**
     * Map child relation keys (FKs pointing to child records)
     */
    private function mapChildRelationKeys(array $data, array $childElements): array {
        $mapped = $data;

        foreach ($childElements as $childElement) {
            $childTable = $childElement['table'];
            $childRelation = $childElement['relation'] ?? [];

            foreach ($childRelation as $childCol => $parentRef) {
                [$parentTable, $parentCol] = explode('.', $parentRef, 2);

                if (isset($mapped[$parentCol])) {
                    $oldValue = $mapped[$parentCol];

                    if (isset($this->idMapping[$childTable][$oldValue])) {
                        $mapped[$parentCol] = $this->idMapping[$childTable][$oldValue];
                    }
                }
            }
        }

        return $mapped;
    }

    /**
     * Check if element has child relations (parent->child FK)
     */
    private function hasChildRelations(array $data, array $childElements): bool {
        foreach ($childElements as $childElement) {
            $childRelation = $childElement['relation'] ?? [];

            foreach ($childRelation as $childCol => $parentRef) {
                [$parentTable, $parentCol] = explode('.', $parentRef, 2);

                if ($parentCol !== 'id' && isset($data[$parentCol])) {
                    return true;
                }
            }
        }

        return false;
    }

    /**
     * Escape table/column identifier
     */
    private function escapeIdentifier(string $identifier): string {
        return '`' . str_replace('`', '``', $identifier) . '`';
    }
}