<?php

/**
 * RecordCopyPaste - Export/Import relational database records via JSON
 *
 * Handles copying records with their relations across databases with same structure.
 * Supports identifier-based duplicate detection, ID mapping, and flexible overrides.
 *
 * Export uses JSON template syntax directly in SQL.
 *
 * @author enured
 * @version 1
 */

namespace IMATHUZH\Qfq\Core\Helper;

use Exception;
use IMATHUZH\Qfq\Core\Database\Database;
use IMATHUZH\Qfq\Core\Report\Link;
use IMATHUZH\Qfq\Core\Store\Session;
use IMATHUZH\Qfq\Core\Store\Store;

class CopyPaste {

    private $db;
    private array $idMapping = [];
    private ?Store $store;
    private string|int|array|null|false $dbIndexData;
    private string|int|array|null|false $dbIndexQfq;
    private mixed $dbArray;
    private mixed $content;

    /**
     * @throws \CodeException
     * @throws \UserFormException
     * @throws \DbException
     * @throws \UserReportException
     */
    public function __construct($content, $db = null, $store = null, $phpUnit = false) {
        if (defined('PHPUNIT_QFQ')) {
            $phpUnit = true;
        }

        if ($store === null) {
            $store = Store::getInstance('', $phpUnit);
        }

        $this->content = trim($content);
        $this->store = $store;

        if ($db === null) {
            $this->dbIndexData = $this->store->getVar(SYSTEM_DB_INDEX_DATA, STORE_SYSTEM);
            $this->dbIndexQfq = $this->store->getVar(SYSTEM_DB_INDEX_QFQ, STORE_SYSTEM);

            $this->dbArray[$this->dbIndexData] = new Database($this->dbIndexData);

            if ($this->dbIndexData != $this->dbIndexQfq) {
                $this->dbArray[$this->dbIndexQfq] = new Database($this->dbIndexQfq);
            }

            $db = $this->dbArray[$this->dbIndexData];
        }

        $this->db = $db;
    }

    /**
     * Handle Export: Parse JSON template or rule names, fetch data, generate complete JSON
     *
     * @return string JSON string for clipboard
     * @throws \UserReportException
     * @throws \Exception
     */
    public function handleExport(): string {
        // Auto-detect: JSON template or rule names
        if (str_starts_with($this->content, '{')) {
            // It's a JSON template
            $template = json_decode($this->content, true);

            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new \UserReportException("Invalid JSON template: " . json_last_error_msg());
            }
        } else {
            // It's comma-separated rule names
            $ruleNames = array_map('trim', explode(',', $this->content));
            $template = $this->loadRules($ruleNames);
        }

        if (!isset($template['element']) || !is_array($template['element'])) {
            throw new \Exception("Invalid template: missing element array");
        }

        // Build result with version and timestamp
        $result = [
            'version' => '1.0',
            'exportedAt' => date('c'),
            'element' => []
        ];

        // Process each root element
        foreach ($template['element'] as $elementTemplate) {
            $processedElements = $this->processElementTemplate($elementTemplate, null);
            $result['element'] = array_merge($result['element'], $processedElements);
        }

        return json_encode($result, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    }

    /**
     * Handle Import: Parse JSON + overrides, check duplicates, insert/update
     *
     * @param string|null $importOverrides Format: "Person.id:345;Address.city:Basel;..."
     * @return array Result with status and created/updated IDs
     * @throws Exception
     */
    public function handleImport(?string $importOverrides = null): array {
        $data = json_decode($this->content, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new Exception("Invalid JSON: " . json_last_error_msg());
        }

        if (!isset($data['element'])) {
            throw new Exception("Invalid JSON structure: missing element array");
        }

        // Parse import overrides
        $overrides = [];
        if ($importOverrides) {
            $overrides = $this->parseImportOverrides($importOverrides);
        }

        // Reset ID mapping for this import
        $this->idMapping = [];

        // Transactions are handled by sql() internally - no manual transaction needed
        try {
            $results = [
                'success' => true,
                'imported' => [],
                'idMapping' => []
            ];

            // Process all root elements recursively
            foreach ($data['element'] as $element) {
                $importResult = $this->importElementRecursive($element, $overrides);
                $results['imported'][] = $importResult;
            }

            $results['idMapping'] = $this->idMapping;

            return $results;

        } catch (Exception $e) {
            // sql() handles rollback internally
            throw $e;
        }
    }

    /**
     * Load and merge multiple rule definitions from CopyPasteRule table
     *
     * @param array $ruleNames Array of rule names
     * @return array Merged template with all elements
     * @throws Exception
     */
    private function loadRules(array $ruleNames): array {
        if (empty($ruleNames)) {
            throw new Exception("No rule names provided");
        }

        $mergedTemplate = [
            'element' => []
        ];

        foreach ($ruleNames as $ruleName) {
            if (empty(trim($ruleName))) continue;

            $query = "SELECT definition FROM CopyPasteRule WHERE name = ? LIMIT 1";
            $result = $this->db->sql($query, ROW_EXPECT_GE_1, [$ruleName]);

            if (empty($result)) {
                throw new Exception("Rule not found: {$ruleName}");
            }

            $row = $result[0];

            // Parse rule definition
            $ruleDefinition = $row['definition'];

            $ruleTemplate = json_decode($ruleDefinition, true);

            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new Exception("Invalid JSON in rule '{$ruleName}': " . json_last_error_msg());
            }

            // Merge elements
            if (isset($ruleTemplate['element']) && is_array($ruleTemplate['element'])) {
                $mergedTemplate['element'] = array_merge(
                    $mergedTemplate['element'],
                    $ruleTemplate['element']
                );
            }
        }

        return $mergedTemplate;
    }

    /**
     * Process element template: fetch data and process child elements (BATCH-OPTIMIZED)
     *
     * @param array $template Element template with table, condition, etc.
     * @param array|null $parentData Parent record data for relation mapping
     * @return array Array of processed elements (can be multiple if condition matches multiple records)
     */
    private function processElementTemplate(array $template, ?array $parentData): array {
        $table = $template['table'] ?? null;
        $dataTemplate = $template['data'] ?? null;
        $condition = $template['condition'] ?? null;
        $relation = $template['relation'] ?? null;
        $identifier = $template['identifier'] ?? null;
        $childTemplates = $template['element'] ?? [];

        if (!$table) {
            throw new Exception("Element template missing table");
        }

        // Fetch records for this level
        $records = $this->fetchRecordsForTemplate($table, $dataTemplate, $condition, $relation, $parentData);

        if (empty($records)) {
            return [];
        }

        // Process child elements in BATCHES
        $childElementsByTemplate = [];

        if (!empty($childTemplates)) {
            // Group child templates by (table, relation, condition) for batching
            foreach ($childTemplates as $childIndex => $childTemplate) {
                $batchKey = $this->getBatchKey($childTemplate);

                // Fetch all children for this template in ONE batch query
                $allChildRecords = $this->fetchChildrenBatch($childTemplate, $records);

                // Group children by parent ID
                $childrenByParent = $this->groupChildrenByParent($allChildRecords, $childTemplate['relation'] ?? null);

                $childElementsByTemplate[$childIndex] = $childrenByParent;
            }
        }

        // Build elements with their children
        $elements = [];

        foreach ($records as $record) {
            $element = [
                'table' => $table,
                'data' => $record,
                'relation' => $relation,
                'identifier' => $identifier,
                'condition' => $condition,
                'element' => []
            ];

            // Assign pre-fetched children to this parent
            foreach ($childTemplates as $childIndex => $childTemplate) {
                $childRecords = $childElementsByTemplate[$childIndex][$record['id']] ?? [];

                if (!empty($childRecords)) {
                    // Recursively process these children's children
                    foreach ($childRecords as $childRecord) {
                        $processedChildren = $this->processChildElementsRecursive(
                            $childTemplate,
                            $childRecord
                        );
                        $element['element'] = array_merge($element['element'], $processedChildren);
                    }
                }
            }

            $elements[] = $element;
        }

        // If multiple records with same template, group data into array
        if (count($elements) > 1 && $this->shouldGroupData($elements)) {
            $grouped = [
                'table' => $table,
                'data' => array_map(fn($e) => $e['data'], $elements),
                'relation' => $relation,
                'identifier' => $identifier,
                'condition' => $condition,
                'element' => $elements[0]['element'] ?? []
            ];
            return [$grouped];
        }

        return $elements;
    }

    /**
     * Fetch records for a template
     */
    private function fetchRecordsForTemplate(string $table, ?array $dataTemplate, ?string $condition, ?array $relation, ?array $parentData): array {
        $whereConditions = [];
        $params = [];

        // Handle relation (FK condition)
        if ($relation && $parentData) {
            foreach ($relation as $childCol => $parentRef) {
                [$parentTable, $parentCol] = explode('.', $parentRef, 2);
                $parentValue = $parentData[$parentCol] ?? null;

                if ($parentValue !== null) {
                    $whereConditions[] = $this->escapeIdentifier($childCol) . " = ?";
                    $params[] = $parentValue;
                }
            }
        }

        // Handle IDs from data template
        if ($dataTemplate && !is_array($dataTemplate)) {
            $dataTemplate = [$dataTemplate];
        }

        if ($dataTemplate && isset($dataTemplate[0]['id'])) {
            $ids = array_map(fn($d) => $d['id'], $dataTemplate);
            $ids = array_filter($ids); // Remove nulls

            if (!empty($ids)) {
                $placeholders = implode(',', array_fill(0, count($ids), '?'));
                $whereConditions[] = $this->escapeIdentifier('id') . " IN ({$placeholders})";
                foreach ($ids as $id) {
                    $params[] = $id;
                }
            }
        }

        // Handle condition SQL string
        if ($condition) {
            $whereConditions[] = "({$condition})";
        }

        $query = "SELECT * FROM " . $this->escapeIdentifier($table);
        if (!empty($whereConditions)) {
            $query .= " WHERE " . implode(' AND ', $whereConditions);
        }

        $result = $this->db->sql($query, ROW_REGULAR, $params);

        return $result;
    }

    /**
     * Fetch children for multiple parents in ONE batch query
     */
    private function fetchChildrenBatch(array $childTemplate, array $parentRecords): array {
        if (empty($parentRecords)) {
            return [];
        }

        $table = $childTemplate['table'] ?? null;
        $condition = $childTemplate['condition'] ?? null;
        $relation = $childTemplate['relation'] ?? null;

        if (!$table || !$relation) {
            return [];
        }

        // Collect all parent IDs
        $parentIds = [];
        foreach ($relation as $childCol => $parentRef) {
            [$parentTable, $parentCol] = explode('.', $parentRef, 2);

            foreach ($parentRecords as $parent) {
                if (isset($parent[$parentCol])) {
                    $parentIds[] = $parent[$parentCol];
                }
            }
        }

        $parentIds = array_unique($parentIds);

        if (empty($parentIds)) {
            return [];
        }

        // Build batch query
        $whereConditions = [];
        $params = [];

        // FK condition with IN clause
        foreach ($relation as $childCol => $parentRef) {
            $placeholders = implode(',', array_fill(0, count($parentIds), '?'));
            $whereConditions[] = $this->escapeIdentifier($childCol) . " IN ({$placeholders})";
            foreach ($parentIds as $id) {
                $params[] = $id;
            }
            break; // Only process first relation entry for batching
        }

        // Additional condition
        if ($condition) {
            $whereConditions[] = "({$condition})";
        }

        $query = "SELECT * FROM " . $this->escapeIdentifier($table);
        if (!empty($whereConditions)) {
            $query .= " WHERE " . implode(' AND ', $whereConditions);
        }

        $result = $this->db->sql($query, ROW_REGULAR, $params);

        return $result;
    }

    /**
     * Group child records by parent ID
     */
    private function groupChildrenByParent(array $childRecords, ?array $relation): array {
        if (empty($childRecords) || !$relation) {
            return [];
        }

        $grouped = [];

        // Get the FK column name from relation
        $fkColumn = null;
        foreach ($relation as $childCol => $parentRef) {
            $fkColumn = $childCol;
            break;
        }

        if (!$fkColumn) {
            return [];
        }

        // Group by FK value
        foreach ($childRecords as $record) {
            $parentId = $record[$fkColumn] ?? null;
            if ($parentId !== null) {
                $grouped[$parentId][] = $record;
            }
        }

        return $grouped;
    }

    /**
     * Process child elements recursively (for nested children)
     */
    private function processChildElementsRecursive(array $template, array $recordData): array {
        $childTemplates = $template['element'] ?? [];

        $element = [
            'table' => $template['table'],
            'data' => $recordData,
            'relation' => $template['relation'] ?? null,
            'identifier' => $template['identifier'] ?? null,
            'condition' => $template['condition'] ?? null,
            'element' => []
        ];

        // Process nested children (recursive)
        if (!empty($childTemplates)) {
            foreach ($childTemplates as $childTemplate) {
                $processedChildren = $this->processElementTemplate($childTemplate, $recordData);
                $element['element'] = array_merge($element['element'], $processedChildren);
            }
        }

        return [$element];
    }

    /**
     * Get batch key for grouping child templates
     */
    private function getBatchKey(array $template): string {
        return json_encode([
            'table' => $template['table'] ?? null,
            'relation' => $template['relation'] ?? null,
            'condition' => $template['condition'] ?? null
        ]);
    }

    /**
     * Determine if multiple elements should be grouped with array data
     */
    private function shouldGroupData(array $elements): bool {
        if (count($elements) <= 1) {
            return false;
        }

        // Check if all elements have same relation, identifier, and condition
        $first = $elements[0];
        $firstRelation = json_encode($first['relation']);
        $firstIdentifier = $first['identifier'];
        $firstCondition = $first['condition'];

        foreach ($elements as $element) {
            if (json_encode($element['relation']) !== $firstRelation ||
                $element['identifier'] !== $firstIdentifier ||
                $element['condition'] !== $firstCondition) {
                return false;
            }
        }

        return true;
    }

    /**
     * Parse import overrides string
     * Format: "Person.id:345;Address.city:Basel;..."
     */
    private function parseImportOverrides(string $definition): array {
        $overrides = [];
        $parts = explode(';', $definition);

        foreach ($parts as $part) {
            if (empty(trim($part))) continue;

            $colonParts = explode(':', $part, 2);
            if (count($colonParts) !== 2) continue;

            [$tableCol, $value] = $colonParts;
            [$table, $column] = explode('.', $tableCol, 2);

            if (!isset($overrides[$table])) {
                $overrides[$table] = [];
            }

            $overrides[$table][$column] = $value;
        }

        return $overrides;
    }

    /**
     * Import element recursively (DEPTH-FIRST)
     *
     * Import order:
     * 1. Map parent relations (FKs to parent records)
     * 2. Import children first (recursively)
     * 3. Map child relations (FKs to child records)
     * 4. Identifier check with all FKs mapped
     * 5. Insert or use existing record
     */
    private function importElementRecursive(array $element, array $overrides): array {
        $table = $element['table'];
        $dataInput = $element['data'];
        $identifier = $element['identifier'] ?? null;
        $relation = $element['relation'] ?? [];

        $result = [
            'table' => $table,
            'items' => []
        ];

        // Handle data as array or single object
        $dataArray = [];
        if (isset($dataInput[0]) && is_array($dataInput[0])) {
            $dataArray = $dataInput;
        } else {
            $dataArray = [$dataInput];
        }

        // Process each data record
        foreach ($dataArray as $data) {
            $oldId = $data['id'] ?? null;

            // Apply overrides
            if (isset($overrides[$table])) {
                $data = array_merge($data, $overrides[$table]);
            }

            // Step 1: Map PARENT relation keys (from parent's idMapping)
            $data = $this->mapRelationKeys($data, $relation);

            // Step 2: Import CHILDREN first (depth-first!)
            $childResults = [];
            if (!empty($element['element'])) {
                foreach ($element['element'] as $childElement) {
                    $childResult = $this->importElementRecursive($childElement, $overrides);
                    $childResults[] = $childResult;
                }
            }

            // Step 3: Map CHILD relation keys (from children's idMapping)
            // This ensures FKs pointing to children use the new IDs
            $data = $this->mapChildRelationKeys($data, $element['element'] ?? []);

            // Step 4: Remove id from data
            unset($data['id']);

            $itemResult = [
                'action' => null,
                'oldId' => $oldId,
                'newId' => null
            ];

            // Step 5: Check for existing record based on identifier
            // Now all FKs (parent and child) are mapped to new IDs!
            if ($identifier !== null && $identifier !== '') {
                $existingId = $this->findByIdentifier($table, $data, $identifier, $relation);

                if ($existingId) {
                    $itemResult['action'] = 'existing';
                    $itemResult['newId'] = $existingId;
                } else {
                    $newId = $this->insertRecord($table, $data);
                    $itemResult['action'] = 'inserted';
                    $itemResult['newId'] = $newId;
                }
            } else {
                // No identifier - always insert
                $newId = $this->insertRecord($table, $data);
                $itemResult['action'] = 'inserted';
                $itemResult['newId'] = $newId;
            }

            // Map old ID to new ID
            if ($oldId) {
                $this->idMapping[$table][$oldId] = $itemResult['newId'];
            }

            $result['items'][] = $itemResult;

            // Add child results
            if (!empty($childResults)) {
                $result['children'] = $childResults;
            }
        }

        return $result;
    }

    /**
     * Find existing record by identifier columns
     */
    private function findByIdentifier(string $table, array $data, string $identifier, array $relation = []): ?int {
        $tableName = $this->escapeIdentifier($table);

        // Determine which columns to use for identification
        $identifierColumns = [];
        if ($identifier === '*') {
            // Use all columns EXCEPT: id, created, modified, and relation FK columns
            $excludeColumns = ['id', 'created', 'modified'];

            // Add relation FK columns to exclude
            foreach ($relation as $fkColumn => $parentRef) {
                $excludeColumns[] = $fkColumn;
            }

            $identifierColumns = array_diff(array_keys($data), $excludeColumns);
        } else {
            // Use specified columns (separated by -)
            $identifierColumns = array_map('trim', explode('-', $identifier));
        }

        if (empty($identifierColumns)) {
            return null;
        }

        // Build WHERE clause
        $conditions = [];
        $params = [];

        foreach ($identifierColumns as $column) {
            if (!array_key_exists($column, $data)) {
                continue;
            }

            $escapedColumn = $this->escapeIdentifier($column);
            $value = $data[$column];

            if ($value === null) {
                $conditions[] = "{$escapedColumn} IS NULL";
            } else {
                $conditions[] = "{$escapedColumn} = ?";
                $params[] = $value;
            }
        }

        if (empty($conditions)) {
            return null;
        }

        $query = "SELECT id FROM {$tableName} WHERE " . implode(' AND ', $conditions) . " LIMIT 1";

        $result = $this->db->sql($query, ROW_REGULAR, $params);

        if (empty($result)) {
            return null;
        }

        return $result[0]['id'] ?? null;
    }

    /**
     * Insert record into database
     */
    private function insertRecord(string $table, array $data): int {
        if (empty($data)) {
            throw new Exception("Cannot insert empty data into {$table}");
        }

        $tableName = $this->escapeIdentifier($table);
        $columns = array_keys($data);
        $values = array_values($data);

        $columnsList = implode(', ', array_map(fn($c) => $this->escapeIdentifier($c), $columns));
        $placeholders = implode(', ', array_fill(0, count($values), '?'));

        $query = "INSERT INTO {$tableName} ({$columnsList}) VALUES ({$placeholders})";

        // sql() returns the insert ID directly for INSERT statements
        $insertId = $this->db->sql($query, ROW_REGULAR, $values);

        if (!$insertId) {
            throw new Exception("Insert failed for {$table}: could not retrieve insert ID");
        }

        return $insertId;
    }

    /**
     * Map relation keys from old IDs to new IDs (PARENT relations)
     * Relation format: { "pId": "Person.id" }
     */
    private function mapRelationKeys(array $data, array $relation): array {
        $mapped = $data;

        foreach ($relation as $column => $reference) {
            // Parse reference: "Person.id"
            [$referencesTable, $referencesColumn] = explode('.', $reference, 2);
            $oldValue = $data[$column] ?? null;

            if ($oldValue !== null && isset($this->idMapping[$referencesTable][$oldValue])) {
                $mapped[$column] = $this->idMapping[$referencesTable][$oldValue];
            }
        }

        return $mapped;
    }

    /**
     * Map child relation keys (FKs pointing to child records)
     *
     * Example: AddressAssign has "adrId" pointing to Address (which is a child element)
     * Child relation format: { "id": "AddressAssign.adrId" } means Address.id maps to AddressAssign.adrId
     *
     * @param array $data Current record data
     * @param array $childElements Child element definitions
     * @return array Data with child FK columns mapped to new IDs
     */
    private function mapChildRelationKeys(array $data, array $childElements): array {
        $mapped = $data;

        foreach ($childElements as $childElement) {
            $childTable = $childElement['table'];
            $childRelation = $childElement['relation'] ?? [];

            // Child relation format: { "childCol": "ParentTable.parentCol" }
            // We need to reverse this: find parentCol and map it using childTable's idMapping
            foreach ($childRelation as $childCol => $parentRef) {
                // Parse: "AddressAssign.adrId"
                [$parentTable, $parentCol] = explode('.', $parentRef, 2);

                // If parentCol exists in current data, map it using child's idMapping
                if (isset($mapped[$parentCol])) {
                    $oldValue = $mapped[$parentCol];

                    if (isset($this->idMapping[$childTable][$oldValue])) {
                        $mapped[$parentCol] = $this->idMapping[$childTable][$oldValue];
                    }
                }
            }
        }

        return $mapped;
    }

    /**
     * Escape table/column identifier
     */
    private function escapeIdentifier(string $identifier): string {
        return '`' . str_replace('`', '``', $identifier) . '`';
    }
}